import random
import math
from config import *

class SkirmishEngine:
    def __init__(self):
        # Tracking "Nodes" of combat for QUAD ASSUMING
        self.last_known_resistance_pos = {} 
        
        # Static cover nodes on the map for suppression recovery
        self.cover_nodes = [
            (100, 100), (400, 300), (700, 500), 
            (200, 450), (600, 150), (300, 550)
        ]
    
    def check_obstacle_collision(self, x, y, dx, dy):
        """Check if movement would collide with obstacles and adjust."""
        new_x = x + dx
        new_y = y + dy
        
        for obs in OBSTACLES:
            # Add 5 pixel buffer to unit size
            if (obs["x"] - 5 <= new_x <= obs["x"] + obs["w"] + 5 and 
                obs["y"] - 5 <= new_y <= obs["y"] + obs["h"] + 5):
                # Collision detected, try alternate paths
                # Try diagonal
                if dx != 0 and dy != 0:
                    # Try just horizontal
                    if not (obs["x"] - 5 <= x + dx <= obs["x"] + obs["w"] + 5 and 
                           obs["y"] - 5 <= y <= obs["y"] + obs["h"] + 5):
                        return dx, 0
                    # Try just vertical
                    if not (obs["x"] - 5 <= x <= obs["x"] + obs["w"] + 5 and 
                           obs["y"] - 5 <= y + dy <= obs["y"] + obs["h"] + 5):
                        return 0, dy
                # Last resort: slide around
                if abs(dx) > abs(dy):
                    return 0, dy if dy != 0 else random.choice([-2, 2])
                else:
                    return dx if dx != 0 else random.choice([-2, 2]), 0
        return dx, dy

    # --- QUAD TECHNIQUES ---

    def calculate_banding_effect(self, unit_list, focus_unit):
        """
        BANDING: When one unit is compromised, the weight (morale) 
        of the entire squad shifts instantly as a single unit.
        """
        # Fix: Accessing focus_unit.health instead of dictionary
        impact = -10 if focus_unit.health < 20 else 5
        
        for unit in unit_list:
            # All units 'stick together' via attribute update
            unit.morale += impact
            # Keeping it strictly positive signed long long style
            if unit.morale < 0: 
                unit.morale = 0

    def expand_intel(self, unit_id, enemy_type):
        """
        EXPANDING: Translating 'Resistance' into associated threat sub-nodes.
        """
        intel_nodes = {
            "identity": enemy_type,
            "threat": 0.85,
            "sub_nodes": ["cover_seeking", "aggressive_posture", "seed_detection"]
        }
        return intel_nodes

    def assume_position(self, unit_id, last_x, last_y, velocity_x, velocity_y):
        """
        ASSUMING: Predicting trajectory nodes when visual data is lost.
        """
        assumed_x = last_x + (velocity_x * 3)
        assumed_y = last_y + (velocity_y * 3)
        return assumed_x, assumed_y

    # --- COMBAT & TACTICS ---

    def resolve_firefight(self, attacker, defender, attacker_weapon, defender_weapon):
        """Calculates damage based on weapon type and triggers suppression nodes using Object attributes."""
        # Get weapon damage range
        attacker_dmg_range = WEAPON_DAMAGE.get(attacker_weapon, (5, 15))
        defender_dmg_range = WEAPON_DAMAGE.get(defender_weapon, (5, 15))
        
        # Attacker damages defender
        damage = random.randint(attacker_dmg_range[0], attacker_dmg_range[1])
        defender.health -= damage
        defender.suppression += 15
        
        # Defender counter-attacks
        counter_damage = random.randint(defender_dmg_range[0], defender_dmg_range[1])
        attacker.health -= counter_damage
        attacker.suppression += 10
        
        return damage

    def check_suppression(self, unit):
        """Checks if suppression weight forces a state change to 'Cover Seeking'."""
        # Corrected to access the .suppression attribute directly
        return unit.suppression > 50

    def find_nearest_cover(self, x, y):
        """Finds the nearest coordinate node designated as 'Cover'."""
        nearest = self.cover_nodes[0]
        min_dist = math.sqrt((x - nearest[0])**2 + (y - nearest[1])**2)

        for node in self.cover_nodes:
            dist = math.sqrt((x - node[0])**2 + (y - node[1])**2)
            if dist < min_dist:
                min_dist = dist
                nearest = node
        return nearest

    def get_cover_drift(self, current_x, current_y, target_x, target_y):
        """Calculates the velocity vector to move a unit toward cover."""
        dx = 4 if target_x > current_x else -4
        dy = 4 if target_y > current_y else -4
        
        if abs(target_x - current_x) < 5: dx = 0
        if abs(target_y - current_y) < 5: dy = 0
            
        return dx, dy
