import random
import math
from config import *
from quad_ai import QUAD_Brain

class RebelUnit:
    def __init__(self, uid, x, y):
        # Standardized constructor for QUAD Engine
        self.id = uid
        self.x = x
        self.y = y
        
        # Stats (Using s64 compatible logic) - Buffed for better combat
        self.health = 120  # Increased from 80
        self.morale = 70
        self.suppression = 0
        self.last_shot_frame = 0  # Fire rate cooldown
        
        # QUAD AI Brain
        self.brain = QUAD_Brain(faction="resistance")
        self.brain.aggression.weight = random.uniform(0.5, 0.75)  # Moderately aggressive
        self.brain.fear.weight = random.uniform(0.3, 0.6)         # Some caution
        
        # Guerrilla States: IDLE, AMBUSH, RETREAT, RALLY
        self.state = "IDLE"
        self.velocity_x = 0
        self.velocity_y = 0
        
        # QUAD Memory Nodes
        self.known_combine_locations = []
        self.rally_point = (random.randint(100, WIDTH-100), random.randint(100, HEIGHT-100))

    def update_awareness(self, combine_units):
        """
        EXPANDING: Identifies nearby Combine units as threat nodes.
        """
        self.known_combine_locations = []
        for unit in combine_units:
            dist = self._get_dist(unit.x, unit.y)
            if dist < 180: # Rebel detection radius
                self.known_combine_locations.append((unit.x, unit.y))

    def get_movement(self, skirmish_engine, combine_units):
        """
        The decision logic processed by your Xeon threads.
        Rebels push toward the center to capture the station.
        """
        self.update_awareness(combine_units)
        
        # 1. Tactical Decision Tree - Aggressive assault
        if len(self.known_combine_locations) > 5 and self.health < 30:
            self.state = "RETREAT"
            return self._flee_from_threats()
        
        elif len(self.known_combine_locations) > 0:
            # Find closest target and engage
            closest = min(self.known_combine_locations, 
                         key=lambda p: math.sqrt((self.x - p[0])**2 + (self.y - p[1])**2))
            self.state = "ENGAGE"
            return self._move_to(closest[0], closest[1], speed=3)
        
        else:
            # Push toward the center of the station
            self.state = "ASSAULT"
            return self._move_to(DEFENSE_CENTER[0], DEFENSE_CENTER[1], speed=3)

    def _hit_and_run(self, engine):
        """
        Moves perpendicularly to threats to confuse 'ASSUME' logic.
        """
        target = self.known_combine_locations[0]
        # Calculate a flanking vector
        dx = 3 if target[1] > self.y else -3
        dy = -3 if target[0] > self.x else 3
        return dx, dy

    def _flee_from_threats(self):
        """Sprints away from the average center of threat nodes."""
        avg_x = sum(p[0] for p in self.known_combine_locations) / len(self.known_combine_locations)
        avg_y = sum(p[1] for p in self.known_combine_locations) / len(self.known_combine_locations)
        
        dx = -5 if avg_x > self.x else 5
        dy = -5 if avg_y > self.y else 5
        return dx, dy

    def _move_to(self, tx, ty, speed):
        """Basic navigation toward a coordinate node."""
        dx = speed if tx > self.x else -speed
        dy = speed if ty > self.y else -speed
        
        if abs(tx - self.x) < 10: dx = 0
        if abs(ty - self.y) < 10: dy = 0
        return dx, dy

    def _get_dist(self, tx, ty):
        return math.sqrt((self.x - tx)**2 + (self.y - ty)**2)

    def take_damage(self, amount):
        self.health -= amount
        if self.health < 0: self.health = 0
