import tkinter as tk
import time
import random
import math
from concurrent.futures import ThreadPoolExecutor
from queue import Queue

# Importing your custom modules
from config import *
from tkinter_map import PatrolMap
from combat_log import CombatLogger
from combine_overwatch import EliteUnit
from civil_protection import CivilProtection
from resistance import RebelUnit
from skirmish import SkirmishEngine
from muzzle_flash import FXEngine
from quad_ai import TeamPersonality

class QUAD_CombatPatrol_Engine:
    def __init__(self, root):
        self.root = root
        self.root.title("C.O. PATROL - SECTOR 17 SIMULATION v0.4 [24-THREAD XEON ACTIVE] - QUAD AI LEARNING")
        
        # 1. Initialize Core Systems
        self.view = PatrolMap(root)
        self.logger = CombatLogger()
        
        # STEP 10: Team Personalities
        self.combine_personality = TeamPersonality("Combine Forces")
        self.resistance_personality = TeamPersonality("Resistance")
        self.skirmish = SkirmishEngine()
        self.fx = FXEngine(self.view.canvas)
        
        # Explicitly using your 24 threads for the Xeon
        self.executor = ThreadPoolExecutor(max_workers=NUM_THREADS)
        
        # 2. Unit Storage - Sized for 24-core parallel processing
        # Elites spawn in defensive formation around center
        self.elites = []
        for i in range(8):
            angle = (i / 8) * 2 * math.pi
            x = DEFENSE_CENTER[0] + math.cos(angle) * (DEFENSE_RADIUS * 0.6)
            y = DEFENSE_CENTER[1] + math.sin(angle) * (DEFENSE_RADIUS * 0.6)
            self.elites.append(EliteUnit(i, x, y))
        
        # Civil Protection patrol around elites
        self.cps = []
        for i in range(20):
            angle = (i / 20) * 2 * math.pi
            x = DEFENSE_CENTER[0] + math.cos(angle) * DEFENSE_RADIUS
            y = DEFENSE_CENTER[1] + math.sin(angle) * DEFENSE_RADIUS
            self.cps.append(CivilProtection(i + 10, x, y))
        
        # Resistance spawns from entrances in waves
        self.rebels = []
        self.spawn_wave_counter = 0
        self.next_wave_time = time.time() + 3  # First wave in 3 seconds
        self.frame_counter = 0  # Track frames for fire rate
        
        # Combine reinforcement tracking
        self.next_elite_spawn = time.time() + 8
        self.next_cp_spawn = time.time() + 6
        self.elite_id_counter = 100
        self.cp_id_counter = 1000
        
        # 3. Draw initial states on Tkinter Canvas
        self.setup_canvas()
        
        # 4. Start the high-performance loop
        self.update_sim()

    def setup_canvas(self):
        """Initial rendering of all units on the map."""
        for e in self.elites:
            self.view.draw_unit(e.id, e.x, e.y, COLOR_COMBINE_BODY, COLOR_COMBINE_OUTLINE)
        for cp in self.cps:
            self.view.draw_unit(cp.id, cp.x, cp.y, COLOR_CP_BODY, COLOR_CP_OUTLINE)

    def spawn_resistance_wave(self):
        """Spawn resistance fighters from the entrances."""
        spawn_point = random.choice(RESISTANCE_SPAWNS)
        num_rebels = random.randint(4, 6)  # 4-6 rebels per wave
        for i in range(num_rebels):
            rebel_id = 100 + self.spawn_wave_counter
            x = spawn_point["x"] + random.randint(-20, 20)
            y = spawn_point["y"] + random.randint(-20, 20)
            rebel = RebelUnit(rebel_id, x, y)
            # STEP 10: New units learn from team personality
            self.resistance_personality.influence_new_unit(rebel.brain)
            self.rebels.append(rebel)
            self.view.draw_unit(rebel.id, rebel.x, rebel.y, COLOR_RESISTANCE_BODY, COLOR_RESISTANCE_OUTLINE)
            self.spawn_wave_counter += 1
    
    def spawn_elite_reinforcement(self):
        """Spawn Elite unit reinforcements."""
        spawn_point = random.choice(COMBINE_SPAWNS)
        num_elites = random.randint(1, 2)  # 1-2 elites
        for i in range(num_elites):
            elite_id = self.elite_id_counter
            self.elite_id_counter += 1
            x = spawn_point["x"] + random.randint(-15, 15)
            y = spawn_point["y"] + random.randint(-15, 15)
            elite = EliteUnit(elite_id, x, y)
            # STEP 10: New elites learn from team
            self.combine_personality.influence_new_unit(elite.brain)
            self.elites.append(elite)
            self.view.draw_unit(elite.id, elite.x, elite.y, COLOR_COMBINE_BODY, COLOR_COMBINE_OUTLINE)
    
    def spawn_cp_reinforcement(self):
        """Spawn Civil Protection reinforcements."""
        spawn_point = random.choice(COMBINE_SPAWNS)
        num_cp = random.randint(3, 5)  # 3-5 CP units
        for i in range(num_cp):
            cp_id = self.cp_id_counter
            self.cp_id_counter += 1
            x = spawn_point["x"] + random.randint(-25, 25)
            y = spawn_point["y"] + random.randint(-25, 25)
            cp = CivilProtection(cp_id, x, y)
            # STEP 10: New CP learn from team
            self.combine_personality.influence_new_unit(cp.brain)
            self.cps.append(cp)
            self.view.draw_unit(cp.id, cp.x, cp.y, COLOR_CP_BODY, COLOR_CP_OUTLINE)

    def process_unit_batch(self, units):
        """
        WORKER THREAD LOGIC: Processes a batch of units.
        Handles QUAD logic: ASSUMING, BANDING, and SUPPRESSION + Visual FX queuing.
        """
        results = []
        all_combine = self.elites + self.cps
        
        for unit in units:
            shots_to_fire = []
            
            # A. Suppression Check (Logic from Skirmish)
            if hasattr(unit, 'suppression') and self.skirmish.check_suppression(unit):
                target_x, target_y = self.skirmish.find_nearest_cover(unit.x, unit.y)
                dx, dy = self.skirmish.get_cover_drift(unit.x, unit.y, target_x, target_y)
                unit.suppression = max(0, unit.suppression - 1)
            
            # B. Tactical Movement & Combat
            elif isinstance(unit, RebelUnit):
                dx, dy = unit.get_movement(self.skirmish, all_combine)
                # Check collision
                dx, dy = self.skirmish.check_obstacle_collision(unit.x, unit.y, dx, dy)
                # Check for firefight with Combine forces
                can_shoot = (self.frame_counter - unit.last_shot_frame) >= FIRE_RATE["rebel_smg"]
                if can_shoot:
                    for elite in self.elites:
                        dist = math.dist((unit.x, unit.y), (elite.x, elite.y))
                        if dist < 120 and elite.health > 0:
                            prev_elite_health = elite.health
                            prev_rebel_health = unit.health
                            self.skirmish.resolve_firefight(unit, elite, "rebel_smg", "elite_rifle")
                            # STEP 7: Apply feedback based on hit/miss
                            unit.brain.apply_feedback(hit_enemy=(elite.health < prev_elite_health), 
                                                     got_hit=(unit.health < prev_rebel_health),
                                                     killed_enemy=(elite.health <= 0))
                            elite.brain.apply_feedback(hit_enemy=(unit.health < prev_rebel_health),
                                                      got_hit=(elite.health < prev_elite_health),
                                                      killed_enemy=(unit.health <= 0))
                            shots_to_fire.append((elite.x, elite.y, unit.x, unit.y, True))
                            shots_to_fire.append((unit.x, unit.y, elite.x, elite.y, False))
                            unit.last_shot_frame = self.frame_counter
                            break  # One engagement per update
                    # Also engage Civil Protection
                    if not shots_to_fire:
                        for cp in self.cps:
                            dist = math.dist((unit.x, unit.y), (cp.x, cp.y))
                            if dist < 100 and cp.health > 0:
                                prev_cp_health = cp.health
                                prev_rebel_health = unit.health
                                self.skirmish.resolve_firefight(unit, cp, "rebel_smg", "cp_pistol")
                                # STEP 7: Feedback
                                unit.brain.apply_feedback(hit_enemy=(cp.health < prev_cp_health),
                                                         got_hit=(unit.health < prev_rebel_health),
                                                         killed_enemy=(cp.health <= 0))
                                cp.brain.apply_feedback(hit_enemy=(unit.health < prev_rebel_health),
                                                       got_hit=(cp.health < prev_cp_health),
                                                       killed_enemy=(unit.health <= 0))
                                shots_to_fire.append((cp.x, cp.y, unit.x, unit.y, True))
                                shots_to_fire.append((unit.x, unit.y, cp.x, cp.y, False))
                                unit.last_shot_frame = self.frame_counter
                                break
            
            elif isinstance(unit, EliteUnit):
                dx, dy = unit.get_movement(self.skirmish, self.rebels)
                # Check collision
                dx, dy = self.skirmish.check_obstacle_collision(unit.x, unit.y, dx, dy)
                # Elites engage rebels aggressively
                can_shoot = (self.frame_counter - unit.last_shot_frame) >= FIRE_RATE["elite_rifle"]
                if can_shoot:
                    for rebel in self.rebels:
                        dist = math.dist((unit.x, unit.y), (rebel.x, rebel.y))
                        if dist < 180 and rebel.health > 0:
                            self.skirmish.resolve_firefight(unit, rebel, "elite_rifle", "rebel_smg")
                            shots_to_fire.append((unit.x, unit.y, rebel.x, rebel.y, True))
                            shots_to_fire.append((rebel.x, rebel.y, unit.x, unit.y, False))
                            unit.last_shot_frame = self.frame_counter
                            break  # One engagement per update
                
            elif isinstance(unit, CivilProtection):
                dx, dy = unit.patrol_logic()
                # Check collision
                dx, dy = self.skirmish.check_obstacle_collision(unit.x, unit.y, dx, dy)
                # CP engages rebels - now equal combatants
                can_shoot = (self.frame_counter - unit.last_shot_frame) >= FIRE_RATE["cp_pistol"]
                if can_shoot:
                    for rebel in self.rebels:
                        dist = math.dist((unit.x, unit.y), (rebel.x, rebel.y))
                        if dist < 120 and rebel.health > 0:  # Same range as rebels (was 110)
                            self.skirmish.resolve_firefight(unit, rebel, "cp_pistol", "rebel_smg")
                            shots_to_fire.append((unit.x, unit.y, rebel.x, rebel.y, True))
                            shots_to_fire.append((rebel.x, rebel.y, unit.x, unit.y, False))
                            unit.last_shot_frame = self.frame_counter
                            break  # One engagement per update
                # C. Reporting/Banding Logic (CP encountering Elite)
                for elite in self.elites:
                    dist = math.dist((unit.x, unit.y), (elite.x, elite.y))
                    if dist < 30:
                        unit.encounter_elite(elite, self.skirmish)
                        self.logger.log_event(int(time.time()), unit.id, 4, elite.id)

            # Apply delta and update the object state
            # Ensure minimum movement to prevent freezing
            if dx == 0 and dy == 0 and isinstance(unit, RebelUnit):
                # Rebels should always move toward center when stuck
                dx = 1 if DEFENSE_CENTER[0] > unit.x else -1
                dy = 1 if DEFENSE_CENTER[1] > unit.y else -1
            
            unit.x += dx
            unit.y += dy
            
            # Decay suppression over time
            if hasattr(unit, 'suppression'):
                unit.suppression = max(0, unit.suppression - 0.5)
            
            # Bound checking (Don't let dots leave the sector)
            unit.x = max(10, min(WIDTH-10, unit.x))
            unit.y = max(10, min(HEIGHT-10, unit.y))
            
            results.append((unit.id, unit.x, unit.y, shots_to_fire))
            
        return results

    def update_sim(self):
        """The main orchestrator loop running on the main thread."""
        start_time = time.time()
        self.frame_counter += 1
        
        # Remove dead units and clean up canvas
        for rebel in self.rebels:
            if rebel.health <= 0:
                try:
                    self.view.canvas.delete(f"ring_{rebel.id}")
                    self.view.canvas.delete(f"body_{rebel.id}")
                except:
                    pass
        for elite in self.elites:
            if elite.health <= 0:
                try:
                    self.view.canvas.delete(f"ring_{elite.id}")
                    self.view.canvas.delete(f"body_{elite.id}")
                except:
                    pass
        for cp in self.cps:
            if cp.health <= 0:
                try:
                    self.view.canvas.delete(f"ring_{cp.id}")
                    self.view.canvas.delete(f"body_{cp.id}")
                except:
                    pass
        
        self.rebels = [r for r in self.rebels if r.health > 0]
        self.elites = [e for e in self.elites if e.health > 0]
        self.cps = [cp for cp in self.cps if cp.health > 0]
        
        # DYNAMIC REINFORCEMENT SYSTEM
        current_time = time.time()
        
        # Spawn resistance waves - more aggressive when winning
        if current_time >= self.next_wave_time:
            if len(self.rebels) < MAX_REBELS:
                # Spawn more rebels if Combine has advantage
                if len(self.elites) + len(self.cps) > len(self.rebels):
                    self.spawn_resistance_wave()
                elif len(self.rebels) < MIN_REBELS:
                    self.spawn_resistance_wave()
            self.next_wave_time = current_time + random.uniform(*REBEL_SPAWN_INTERVAL)
        
        # Spawn Elite reinforcements when losing ground
        if current_time >= self.next_elite_spawn:
            if len(self.elites) < MAX_ELITES:
                # More likely to spawn if rebels outnumber or elites are low
                if len(self.elites) < MIN_ELITES or len(self.rebels) > len(self.elites) + len(self.cps):
                    self.spawn_elite_reinforcement()
            self.next_elite_spawn = current_time + random.uniform(*ELITE_SPAWN_INTERVAL)
        
        # Spawn CP reinforcements to maintain defense
        if current_time >= self.next_cp_spawn:
            if len(self.cps) < MAX_CP:
                # Spawn if CP numbers are low or resistance is strong
                if len(self.cps) < MIN_CP or len(self.rebels) > 15:
                    self.spawn_cp_reinforcement()
            self.next_cp_spawn = current_time + random.uniform(*CP_SPAWN_INTERVAL)
        
        # 1. Prepare unit list and partition for 24 threads
        all_units = self.elites + self.cps + self.rebels
        chunk_size = max(1, len(all_units) // NUM_THREADS)
        chunks = [all_units[i:i + chunk_size] for i in range(0, len(all_units), chunk_size)]

        # 2. Dispatch to Xeon ThreadPool
        # Each thread handles a 'Squad' of units to minimize overhead
        futures = [self.executor.submit(self.process_unit_batch, chunk) for chunk in chunks]

        # 3. Collect results and update the Tkinter View + Trigger FX
        for future in futures:
            batch_data = future.result()
            for uid, nx, ny, shots in batch_data:
                self.view.update_position(uid, nx, ny)
                # Render muzzle flashes and regal tracers
                for sx, sy, tx, ty, is_combine in shots:
                    self.fx.fire_shot(sx, sy, tx, ty, is_combine)

        # 4. Schedule next frame (30 FPS targeting)
        # Keeping you fresh and the sim smooth
        elapsed = (time.time() - start_time) * 1000
        delay = max(1, int(33 - elapsed))
        self.root.after(delay, self.update_sim)

if __name__ == "__main__":
    root = tk.Tk()
    sim = QUAD_CombatPatrol_Engine(root)
    root.mainloop()
