import random
import math
from config import *
from quad_ai import QUAD_Brain

class EliteUnit:
    def __init__(self, uid, x, y):
        # Standardized constructor for the QUAD Engine batch processor
        self.id = uid
        self.x = x
        self.y = y
        
        # Regal Stats (Compatible with signed long long int logic)
        self.health = 180  # Elites are SUPREME commandos (was 150)
        self.morale = 100
        self.suppression = 0
        self.last_shot_frame = 0  # Fire rate cooldown
        
        # QUAD AI Brain
        self.brain = QUAD_Brain(faction="combine_elite")
        self.brain.aggression.weight = random.uniform(0.6, 0.85)  # Elites are aggressive
        self.brain.accuracy.weight = random.uniform(0.7, 0.9)     # Highly accurate
        
        # QUAD v0.4 Node Storage & State Management
        self.memory_nodes = [] 
        self.active_target = None
        self.state = "PATROL" # States: PATROL, SKIRMISH, SEEK_COVER

    def get_movement(self, skirmish_engine, rebel_list):
        """
        Decision matrix running on a dedicated thread.
        Elite commandos use advanced tactics: cover, flanking, coordinated strikes.
        """
        
        # 1. TACTICAL OVERRIDE: Suppression (Stress Response)
        if skirmish_engine.check_suppression(self):
            self.state = "SEEK_COVER"
            target_x, target_y = skirmish_engine.find_nearest_cover(self.x, self.y)
            return skirmish_engine.get_cover_drift(self.x, self.y, target_x, target_y)

        # 2. COMMANDO TACTICS: Find nearest threat
        nearest_rebel = None
        min_dist = float('inf')
        for rebel in rebel_list:
            if rebel.health > 0:
                dist = math.dist((self.x, self.y), (rebel.x, rebel.y))
                if dist < min_dist:
                    min_dist = dist
                    nearest_rebel = rebel
        
        if nearest_rebel:
            # A. Close range - AGGRESSIVE RUSH AND FLANK
            if min_dist < ELITE_RUSH_DISTANCE:
                if random.random() < ELITE_RUSH_CHANCE or min_dist < 80:
                    self.state = "RUSHING"
                    # Flank at an angle instead of straight charge
                    angle_offset = random.choice([-0.5, 0.5])
                    dx_base = nearest_rebel.x - self.x
                    dy_base = nearest_rebel.y - self.y
                    # Add perpendicular component for flanking
                    dx = int(5 * (dx_base/max(1, min_dist)) + angle_offset * (dy_base/max(1, min_dist)))
                    dy = int(5 * (dy_base/max(1, min_dist)) - angle_offset * (dx_base/max(1, min_dist)))
                    return dx, dy
            
            # B. Medium range - TACTICAL ADVANCE with superior positioning
            elif min_dist < 300:
                self.state = "ENGAGE"
                # Move to maintain optimal firing distance (100-150 units)
                if min_dist > 150:
                    dx = 3 if nearest_rebel.x > self.x else -3
                    dy = 3 if nearest_rebel.y > self.y else -3
                else:
                    # Hold position or strafe
                    dx = random.choice([-2, 0, 2])
                    dy = random.choice([-2, 0, 2])
                return dx, dy

        # 3. QUAD ASSUMING: Move toward ghost coordinates from CP reports
        # If we have an assumed target but no direct visual, we hunt the 'Assumption Node'.
        if self.state == "SKIRMISH" and self.active_target:
            tx, ty = self.active_target
            # Calculate vector toward the assumed enemy position
            dx = 2 if tx > self.x else -2
            dy = 2 if ty > self.y else -2
            
            # If we reach the assumption and the resistance is not found, reset
            if math.dist((self.x, self.y), (tx, ty)) < 15:
                print(f"[OVERWATCH ELITE {self.id}] Assumption clear. Resuming Sector Patrol.")
                self.state = "PATROL"
                self.active_target = None
            return dx, dy

        # 3. NO THREATS - PATROL: Return to defensive positions
        self.state = "PATROL"
        dx_to_center = DEFENSE_CENTER[0] - self.x
        dy_to_center = DEFENSE_CENTER[1] - self.y
        dist_to_center = math.sqrt(dx_to_center**2 + dy_to_center**2)
        
        if dist_to_center > DEFENSE_RADIUS * 0.8:
            return 2 if dx_to_center > 0 else -2, 2 if dy_to_center > 0 else -2
        else:
            return random.choice([-1, 0, 1]), random.choice([-1, 0, 1])

    def receive_report(self, intel_node, skirmish_engine):
        """
        QUAD EXPANDING & ASSUMING:
        The Elite takes a 'Suspicion Node' from Civil Protection and expands it.
        """
        print(f"[OVERWATCH ELITE {self.id}] Processing Intel Node from Civil Protection...")
        
        # QUAD EXPANDING: Translate raw data into threat sub-nodes
        expanded_intel = skirmish_engine.expand_intel(self.id, "Resistance")
        
        # QUAD ASSUMING: Predict trajectory based on the word/location found
        # We assume the rebel is moving at a generic speed of 5 pixels/tick
        predicted_x, predicted_y = skirmish_engine.assume_position(
            self.id, 
            intel_node['pos'][0], 
            intel_node['pos'][1], 
            random.randint(-5, 5), 
            random.randint(-5, 5)
        )
        
        self.active_target = (predicted_x, predicted_y)
        self.state = "SKIRMISH"
        
    def take_damage(self, amount):
        """Standard health reduction using s64 logic."""
        self.health -= amount
        if self.health < 0: self.health = 0
